document.addEventListener('DOMContentLoaded', () => {
  const latInput = document.getElementById('latitude');
  const lonInput = document.getElementById('longitude');
  const saveButton = document.getElementById('saveButton');
  const statusDiv = document.getElementById('status');
  
  // Elemen Pencarian
  const searchInput = document.getElementById('search-input');
  const searchBtn = document.getElementById('search-btn');

  const DEFAULT_LAT = -3.3193; // Default Banjarmasin Kota
  const DEFAULT_LON = 114.5898;
  
  let map;
  let marker;

  // --- Fungsi Update Marker & Input ---
  // Digunakan agar peta dan inputan selalu sinkron
  function updatePosition(lat, lon, updateMapCenter = false) {
    const newLat = parseFloat(lat);
    const newLon = parseFloat(lon);

    if (isNaN(newLat) || isNaN(newLon)) return;

    // Pindahkan marker
    if (marker) {
      marker.setLatLng([newLat, newLon]);
    }
    
    // Pindahkan tampilan peta jika diminta (misal hasil search)
    if (updateMapCenter && map) {
      map.setView([newLat, newLon], 16); // Zoom lebih dekat (16)
    }

    // Update nilai di kotak input (batasi 6 desimal)
    latInput.value = newLat.toFixed(6);
    lonInput.value = newLon.toFixed(6);
  }

  // --- Inisialisasi Peta ---
  function initMap(lat, lon) {
    map = L.map('map').setView([lat, lon], 13);

    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
      attribution: '&copy; OpenStreetMap contributors'
    }).addTo(map);

    marker = L.marker([lat, lon], {draggable: true}).addTo(map);

    // Event: Marker digeser -> Update Input
    marker.on('drag', function(e) {
      const pos = marker.getLatLng();
      latInput.value = pos.lat.toFixed(6);
      lonInput.value = pos.lng.toFixed(6);
    });

    // Event: Peta diklik -> Pindah Marker & Update Input
    map.on('click', function(e) {
      updatePosition(e.latlng.lat, e.latlng.lng);
    });
  }

  // --- 1. Load Data Awal ---
  chrome.storage.local.get(['fakeLatitude', 'fakeLongitude'], (result) => {
    let currentLat = parseFloat(result.fakeLatitude) || DEFAULT_LAT;
    let currentLon = parseFloat(result.fakeLongitude) || DEFAULT_LON;
    
    // Isi input awal
    latInput.value = currentLat;
    lonInput.value = currentLon;

    initMap(currentLat, currentLon);
  });

  // --- 2. Fitur Pencarian Lokasi (Nominatim API) ---
  searchBtn.addEventListener('click', () => {
    const query = searchInput.value;
    if (!query) return;

    searchBtn.textContent = "Mencari...";
    searchBtn.disabled = true;

    // Request ke OpenStreetMap
    fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(query)}`)
      .then(response => response.json())
      .then(data => {
        if (data && data.length > 0) {
          // Ambil hasil pertama
          const result = data[0];
          updatePosition(result.lat, result.lon, true); // true = pindahkan kamera peta
          statusDiv.textContent = `Ditemukan: ${result.display_name.split(',')[0]}`;
          statusDiv.style.color = 'blue';
        } else {
          statusDiv.textContent = 'Lokasi tidak ditemukan.';
          statusDiv.style.color = 'red';
        }
      })
      .catch(err => {
        console.error(err);
        statusDiv.textContent = 'Gagal mencari lokasi. Cek koneksi.';
        statusDiv.style.color = 'red';
      })
      .finally(() => {
        searchBtn.textContent = "Cari";
        searchBtn.disabled = false;
      });
  });

  // --- 3. Fitur Ketik Manual (Reverse Sync) ---
  // Saat user mengetik di kotak Lat/Lon, marker di peta ikut pindah
  function handleManualInput() {
    const lat = parseFloat(latInput.value);
    const lon = parseFloat(lonInput.value);
    if (!isNaN(lat) && !isNaN(lon) && map && marker) {
      marker.setLatLng([lat, lon]);
      map.panTo([lat, lon]); // Geser peta pelan-pelan ke lokasi baru
    }
  }

  // Dengarkan event 'input' (saat mengetik)
  latInput.addEventListener('input', handleManualInput);
  lonInput.addEventListener('input', handleManualInput);

  // --- 4. Simpan & Aktifkan ---
  saveButton.addEventListener('click', async () => {
    await chrome.storage.local.set({
      fakeLatitude: latInput.value,
      fakeLongitude: lonInput.value,
    });
    
    statusDiv.textContent = 'Lokasi Disimpan! Refresh halaman presensi.';
    statusDiv.style.color = 'green';
    setTimeout(() => { statusDiv.textContent = ''; }, 3000);
  });
  
  // Tekan Enter di kolom pencarian untuk mencari
  searchInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') searchBtn.click();
  });
});